const express = require('express');
const cors = require('cors');
const bodyParser = require('body-parser');
const admin = require('firebase-admin');
const db = require('./db');
const path = require('path');

// Initialize Firebase Admin
admin.initializeApp({
  projectId: 'spidergeni1'
});

const app = express();
const PORT = 3001;


// Middleware
app.use(cors());
app.use(bodyParser.json({ limit: '50mb' }));

// --- Auth Middleware ---
const authenticateUser = async (req, res, next) => {
  const authHeader = req.headers.authorization;
  if (!authHeader || !authHeader.startsWith('Bearer ')) {
    return res.status(401).json({ error: 'Unauthorized: No token provided' });
  }

  const token = authHeader.split(' ')[1];
  try {
    const decodedToken = await admin.auth().verifyIdToken(token);
    req.user = decodedToken;

    // Get/Create Meta
    let meta = db.getMeta(req.user.uid);
    if (!meta) {
      // First user is Admin?
      const allMeta = db.getAllMeta();
      const isFirstUser = Object.keys(allMeta).length === 0;

      meta = {
        email: req.user.email,
        role: isFirstUser ? 'admin' : 'user',
        approved: isFirstUser ? true : false,
        createdAt: new Date().toISOString()
      };
      db.updateMeta(req.user.uid, meta);
    }

    req.user.meta = meta;
    next();
  } catch (error) {
    console.error("Auth Error:", error);
    return res.status(403).json({ error: 'Unauthorized: Invalid token' });
  }
};

const requireApproval = (req, res, next) => {
  if (!req.user || !req.user.meta || !req.user.meta.approved) {
    return res.status(403).json({ error: 'Forbidden: Account not approved.' });
  }
  next();
};

const requireAdmin = (req, res, next) => {
  if (!req.user || !req.user.meta || req.user.meta.role !== 'admin') {
    return res.status(403).json({ error: 'Forbidden: Admins only.' });
  }
  next();
};

// --- Auth Routes ---

app.get('/api/auth/status', authenticateUser, (req, res) => {
  res.json(req.user.meta);
});

// --- Admin Routes ---

app.get('/api/admin/users', authenticateUser, requireAdmin, async (req, res) => {
  try {
    const allMeta = db.getAllMeta();
    // Maybe fetch limits? For now return all.
    res.json(allMeta);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

app.post('/api/admin/approve', authenticateUser, requireAdmin, (req, res) => {
  try {
    const { uid, approved } = req.body;
    db.updateMeta(uid, { approved });
    res.json({ success: true });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// --- API Routes ---

// Get all tasks (rows)
// Get all tasks (rows)
app.get('/api/tasks', authenticateUser, requireApproval, (req, res) => {
  try {
    const data = db.read(req.user.uid);
    res.json(data.tasks || []);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Save/Update tasks (bulk or single)
// Save/Update tasks (bulk or single)
app.post('/api/tasks', authenticateUser, requireApproval, (req, res) => {
  try {
    const { tasks } = req.body; // Expecting array of tasks
    if (!Array.isArray(tasks)) {
      return res.status(400).json({ error: 'Invalid data format. Expected "tasks" array.' });
    }

    // We overwrite the tasks array or merge? 
    // For simplicity in this app (sync functionality), we might replace the list or merge by ID.
    // Let's implement a merge/upsert strategy.

    db.update(req.user.uid, (data) => {
      const existingTasks = data.tasks || [];
      const taskMap = new Map(existingTasks.map(t => [t.id, t]));

      tasks.forEach(task => {
        taskMap.set(task.id, task);
      });

      data.tasks = Array.from(taskMap.values());
    });

    res.json({ success: true, count: tasks.length });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Get Sheets
app.get('/api/sheets', authenticateUser, requireApproval, (req, res) => {
  try {
    const data = db.read(req.user.uid);
    res.json(data.sheets || []);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Save Sheets (Upsert/Replace)
app.post('/api/sheets', authenticateUser, requireApproval, (req, res) => {
  try {
    const sheets = req.body; // Expecting array of sheets
    if (!Array.isArray(sheets)) {
      return res.status(400).json({ error: 'Invalid data format. Expected array of sheets.' });
    }

    db.update(req.user.uid, (data) => {
      data.sheets = sheets;
    });

    res.json({ success: true });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Delete a task
// Delete a task
app.delete('/api/tasks/:id', authenticateUser, requireApproval, (req, res) => {
  try {
    const { id } = req.params;
    db.update(req.user.uid, (data) => {
      data.tasks = (data.tasks || []).filter(t => t.id !== id);
    });
    res.json({ success: true });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Get Settings
// Get Settings
app.get('/api/settings', authenticateUser, requireApproval, (req, res) => {
  try {
    const data = db.read(req.user.uid);
    res.json(data.settings || {});
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Save Settings
// Save Settings
app.post('/api/settings', authenticateUser, requireApproval, (req, res) => {
  try {
    const settings = req.body;
    db.update(req.user.uid, data => {
      data.settings = { ...data.settings, ...settings };
    });
    res.json({ success: true });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Log entry
// Log entry
app.post('/api/logs', authenticateUser, requireApproval, (req, res) => {
  try {
    const logEntry = req.body; // { timestamp, level, message }
    db.update(req.user.uid, data => {
      const logs = data.logs || [];
      logs.push(logEntry);
      if (logs.length > 1000) logs.shift(); // Keep last 1000
      data.logs = logs;
    });
    res.json({ success: true });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

app.get('/api/logs', authenticateUser, requireApproval, (req, res) => {
  try {
    const data = db.read(req.user.uid);
    res.json(data.logs || []);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

app.delete('/api/logs', authenticateUser, requireApproval, (req, res) => {
  try {
    db.update(req.user.uid, data => {
      data.logs = [];
    });
    res.json({ success: true });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Start Server
app.listen(PORT, () => {
  console.log(`SERVER running on http://localhost:${PORT}`);
  console.log(`Global Database Path: ${db.DB_PATH}`);
});
